/*
 * If distributed as part of the Linux kernel, this code is licensed under the
 * terms of the GPL v2.
 *
 * Otherwise, the following license terms apply:
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1) Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2) Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3) The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESSED OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <linux/slab.h>		/* for kmalloc */
#include <linux/interrupt.h>	/* for irq_disable */
#include "fcu_subr.h"		/* for plist_t */

/*
 * num個のsizeバイトなデータをallocateする
 * allocate時はwaitしない
 *  0 : 成功
 * -1 : 失敗
 */
int
plalloc(plist_t *plp, int num, size_t size, long irq)
{
	size_t asize = roundup(size, sizeof(long)); /* サイズは切り上げ */
	size_t total = num * asize;

	memset(plp, 0, sizeof(plist_t));
	plp->st = kmalloc(total, GFP_ATOMIC);
	if (plp->st == NULL)
		return (-1);
	memset(plp->st, 0, total);
	plp->siz = size;
	plp->asz = asize;
	plp->en = plp->st + total;

	plp->hd = plp->tl = plp->st;
	plp->cnt = 0;				/* データ個数 */
	plp->emp = num;				/* 空き個数 */
	plp->sc_irq = irq;				/* disable_irq() */
	return (0);
}


/*
 * plpのエリアをfreeする
 */
void
plfree(plist_t *plp)
{
	if (plp->st)
		kfree(plp->st);
	memset(plp, 0, sizeof(plist_t));
}


/*
 * データを取り出す
 *  0 : ptrへ取り出し成功
 * -1 : データがない
 */
int
plget(plist_t *plp, caddr_t ptr)
{

	if (plp->cnt <= 0) {
		return (-1);
	}
	memcpy(ptr, plp->hd, plp->siz);
	plp->hd += plp->asz;
	if (plp->hd >= plp->en)
		plp->hd = plp->st;
	plp->cnt--;
	plp->emp++;

	return (0);
}

/*
 * データを格納する
 *  0 : ptrのデータを格納成功
 * -1 : 空きがない
 */
int
plput( plist_t *plp, caddr_t ptr)
{

	if (plp->emp <= 0) {
		return (-1);
	}
	memcpy(plp->tl, ptr, plp->siz);
	plp->tl += plp->asz;
	if (plp->tl >= plp->en)
		plp->tl = plp->st;
	plp->cnt++;
	plp->emp--;

	return (0);
}


/*
 * plpを初期化する
 */
void
plinit(plist_t *plp)
{

	plp->hd = plp->tl = plp->st;
	plp->cnt = 0;				/* データ個数 */
	plp->emp = (plp->en - plp->st) / plp->asz; /* 空き個数 */
}

#define  CACHE_LINE_SIZE   (64)
/**
 * キャッシュフラッシュする
 */
static void __cache_flush(unsigned long  addr)
{
//	printk("## addr = %p\n", addr);
	asm volatile ("mcr p15, 0, %0, c7, c10, 1"::"r"(addr));
	return;
}
void  cache_flush_range(unsigned char  *addr, int  size)
{
	unsigned char  *start;
	unsigned char  *end;
	unsigned char  *cur;

	printk("##%s : (%p, %d)\n", __FUNCTION__, addr, size);
	start = (unsigned char *)(((unsigned long)addr) & ~(CACHE_LINE_SIZE-1));
	end   = (unsigned char *)((unsigned long)addr + (unsigned long)size);
	cur   = start;
	while(cur < end)
	{
		__cache_flush((unsigned long)cur);
		cur += CACHE_LINE_SIZE;
	}
	asm volatile("dsb");
	return;
}

/**
 * キャッシュインバリデートする
 */

static void __cache_invalidate(unsigned long  addr)
{
//	printk("## addr = %p\n", addr);
	asm volatile ("mcr p15, 0, %0, c7, c6, 1"::"r"(addr));
	return;
}
void  cache_invalidate_range(unsigned char  *addr, int  size)
{
	unsigned char  *start;
	unsigned char  *end;
	unsigned char  *cur;

	printk("##%s : (%p, %d)\n", __FUNCTION__, addr, size);
	start = (unsigned char *)(((unsigned long)addr) & ~(CACHE_LINE_SIZE-1));
	end   = (unsigned char *)((unsigned long)addr + (unsigned long)size);
	cur   = start;
	while(cur < end)
	{
		__cache_invalidate((unsigned long)cur);
		cur += CACHE_LINE_SIZE;
	}
	asm volatile("dsb");
	return;
}

